//
//  RexselConfiguration.swift
//  CodeEditorView
//
//  Modified by Hugh Field-Richards on 07/01/2025.
//  Created by Manuel M T Chakravarty on 10/01/2023.
//

import Foundation
import RegexBuilder
import LanguageSupport

private let rexselReservedIdentifiers =
[ "stylesheet", "version", "id", "exclude-result-prefixes", "xmlns",
  "apply-imports",
  "apply-templates", "using", "scope",
  "attribute", "namespace",
  "attribute-set", "use-attribute-sets",
  "call", "with",
  "choose", "when", "otherwise",
  "comment",
  "copy", "copy-of",
  "decimalFormat", "decimalSeparator", "groupingSeparator", "infinity", "minusSign", "notNumber", "percent", "perMille", "zeroDigit", "digit", "patternSeparator",
  "element", "namespace",
  "foreach",
  "proc", "parameter",
  "if",
  "include", "import",
  "key", "keyNodes",
  "match", "priority",
  "message", "terminate", "yes", "no",
  "namespace-alias", "map-from", "to",
  "number", "any", "count", "from", "value",  "traditional", "lang", "grouping-separator", "grouping-size",
  "output", "text", "UTF-16", "doctype-system", "doctype-public", "cdata-section-elements", "encoding", "method", "omit-xml-declaration", "standalone", "indent",
  "parameter",
  "preserve-space",
  "processing-instruction",
  "sort", "descending", "lower-first", "lang",
  "strip-space",
  "text", "value", "disable-output-escaping",
  "variable", "constant"
]

private let rexselReservedOperators =
[".", ",", ":", ";", "=", "@", "#", "&", "->", "`", "?", "!"]

extension LanguageConfiguration {
    
    /// Language configuration for Rexsel
    ///
    public static func rexsel(_ languageService: LanguageService? = nil) -> LanguageConfiguration {
        let numberRegex: Regex<Substring> = Regex {
            optNegation
            ChoiceOf {
                Regex { /0b/; binaryLit }
                Regex { /0o/; octalLit }
                Regex { /0x/; hexalLit }
                Regex { /0x/; hexalLit; "."; hexalLit; Optionally { hexponentLit } }
                Regex { decimalLit; "."; decimalLit; Optionally { exponentLit } }
                Regex { decimalLit; exponentLit }
                decimalLit
            }
        }
        let plainIdentifierRegex: Regex<Substring> = Regex {
            identifierHeadCharacters
            ZeroOrMore {
                identifierCharacters
            }
        }
        let identifierRegex = Regex {
            ChoiceOf {
                plainIdentifierRegex
                Regex { "`"; plainIdentifierRegex; "`" }
                Regex { "$"; decimalLit }
                Regex { "$"; plainIdentifierRegex }
            }
        }
        let operatorRegex = Regex {
            ChoiceOf {
                
                Regex {
                    operatorHeadCharacters
                    ZeroOrMore {
                        operatorCharacters
                    }
                }
                
                Regex {
                    "."
                    OneOrMore {
                        CharacterClass(operatorCharacters, .anyOf("."))
                    }
                }
            }
        }
        return LanguageConfiguration( name: "Rexsel",
                                      supportsSquareBrackets: true,
                                      supportsCurlyBrackets: true,
                                      stringRegex: /\"(?:\\\"|[^\"])*+\"/,
                                      characterRegex: nil,
                                      numberRegex: numberRegex,
                                      singleLineComment: "//",
                                      nestedComment: (open: "/*", close: "*/"),
                                      identifierRegex: identifierRegex,
                                      operatorRegex: operatorRegex,
                                      reservedIdentifiers: rexselReservedIdentifiers,
                                      reservedOperators: rexselReservedOperators,
                                      languageService: languageService)
    }
}
