//
//  RunCompiler.swift
//  RexselEditor
//
//  Created by Hugh Field-Richards on 05/11/2024.
//  Copyright 2025 Hugh Field-Richards. All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  https://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.

import Foundation
import RexselKernel
import LanguageSupport

struct RunCompiler {

    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Instance properties
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

    /// The main Rexsel compiler
    var documentCompiler: RexselKernel?

    /// The string that holds the complete source.
    private(set) var sourceString: String

    /// The string that holds the generated listing (destination XSL).
    private(set) var listingString: String = ""

    /// The error list as a block.
    private(set) var errorListingString: String = ""
    
    /// A list of error numbers.
    private var errorNumbersList = [Int]()
    
    /// Insert the Rexsel source line numbers into the XSL
    var insertLineNumbers: Bool
    
    /// Show any undefined symbols as well as normal errors.
    /// This is used when Rexsel/XSL modules are used. It helps
    /// by only showing syntax errors.
    var showUndefined: Bool
    
    /// Inhibit the explicit use of XSL namespaces throughout
    /// the XSL elements.
    ///
    /// In other words use
    /// ```
    ///    xmlns="http://www.w3.org/1999/XSL/Transform"
    /// ```
    /// rather than
    /// ```
    ///    xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    ///    ```
    var defaultXMLNS: Bool
    
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Initialisation Methods
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    //
    /// Initialise RunCompiler with source string.
    ///
    /// - Parameters:
    ///    - sourceString: The source string from the source window
    ///    - using: The compiler instance for this document
    ///    - lineNumbers: Insert line numbers.
    ///    - showUndefined: Show undefined symbols.
    ///    - useDefaultXMLNS: Use default XML namespace.

    init( with sourceString: String,
          using theCompiler: RexselKernel,
          lineNumbers: Bool,
          showUndefined: Bool,
          useDefaultXMLNS: Bool )
    {
        self.sourceString = sourceString
        self.errorNumbersList = []
        self.listingString = ""
        self.errorListingString = ""
        self.documentCompiler = theCompiler
        self.insertLineNumbers = lineNumbers
        self.showUndefined = showUndefined
        self.defaultXMLNS = useDefaultXMLNS
    }

    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    //
    /// Run the Rexsel compiler.
    ///
    /// - Parameters:
    ///    - sourceString: The source string from the source window.
    ///    - lineNumbers: Insert line numbers.
    ///    - showUndefined: Show undefined symbols.
    ///    - useDefaultXMLNS: Use default XML namespace.
    /// - Returns: _codeListing_: the compiled XSL code, _statusListing_: the error listing and symbol table

    mutating func run( with sourceString: String = "",
                       lineNumbers: Bool,
                       showUndefined: Bool,
                       useDefaultXMLNS: Bool ) -> ( codeListing: String,
                                                    statusListing: String,
                                                    errorLines: [Int],
                                                    errorList: [Int],
                                                    errorMessages: Set<TextLocated<Message>> )
    {
        guard let rexselCompiler = documentCompiler else {
            return ( "", "No compiler found", [], [], [] )
        }
        rexselCompiler.source.readIntoCompilerString( sourceString )
        let runResults = rexselCompiler.run( showUndefined: showUndefined,
                                             insertLineNumbers: lineNumbers,
                                             defaultNameSpace: useDefaultXMLNS,
                                             verbose: false )
        
        listingString = "\(runResults.codeListing)"
        errorListingString = "\(runResults.errorListing)\n\(runResults.symbolTable)"
        errorNumbersList = runResults.errors
        // Get all the details of each error.
        var messages: Set<TextLocated<Message>> = Set()
        let sortedList = errorNumbersList.sorted()
        var errorLines = [Int]()
        for errorNumber in sortedList {
            if let errorReport = rexselCompiler.getErrorAt(errorNumber) {
                errorLines.append( errorReport.line )
                let message = Message( category: errorReport.category == 2 ? .warning : .error,
                                       length: 10,
                                       summary: errorReport.message,
                                       description: AttributedString( errorReport.suggestion ) )
                messages.insert( TextLocated(location: TextLocation( oneBasedLine: errorReport.line, column: errorReport.pos ), entity: message) )
            }
        }
        return( listingString, errorListingString, errorLines, errorNumbersList, messages )
    }
}
