//
//  ToolBarView.swift
//  RexselEditor
//
//  Created by Hugh Field-Richards on 04/11/2024.
//  Copyright 2025 Hugh Field-Richards. All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  https://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
//
// This holds the various checkboxes and buttons at the bottom of the panel.

import SwiftUI

struct ToolBarView: View {
    
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Structure Static Properties
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

    private static let dateFormatter: DateFormatter = {
        let formatter = DateFormatter()
        formatter.dateFormat = "dd/MM/yyyy hh:mm:ss"
        return formatter
    }()

    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Structure Property Bindings
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

    /// The editable Rexsel source
    @Binding var sourceDocument: RexselSourceStruct

    /// Include the source line numbers in the XSL output.
    @Binding var insertLineNumbers: Bool
    
    /// Show any undefined symbols as well as normal errors.
    /// This is used when Rexsel/XSL modules are used. It helps
    /// by only showing syntax errors.
    @Binding var showUndefined: Bool
    
    /// Allow auto compile (this can be turned off when large files are used).
    @Binding var autoCompile: Bool
    
    /// Inhibit the explicit use of XSL namespaces throughout
    /// the XSL elements.
    ///
    /// In other words use
    /// ```
    ///    xmlns="http://www.w3.org/1999/XSL/Transform"
    /// ```
    /// rather than
    /// ```
    ///    xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    /// ```
    @Binding var defaultXMLNS: Bool
    
    /// The listing text (XSL) to be placed in the top right panel.
    @Binding var listingText: String
    
    /// Does the source need saving?
    @Binding var isSourceDirty: Bool

    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Structure Instance Properties
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

    var fileName: String {
        return sourceDocument.fileName
    }
    
    /// Current file name (no path)
    var fileNameWithoutExtension: String {
        let fileName = sourceDocument.fileName
        if let dotIndex = fileName.firstIndex(of: ".") {
            return String(fileName[..<dotIndex])
        }
        return "Untitled"
    }
    
    /// Time stamp
    var currentDateTime: String {
        return Self.dateFormatter.string( from: Date() )
    }

    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Structure Instance methods
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    //
    /// Save the XSL in a file.
    ///
    /// - Parameters:
    ///   - compiledXSL: The compiled XSLT held as a string.
    
    func saveXSL( _ compiledXSL: String ) {

        let prefix = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
        let header = "Produced using Rexsel compiler on \(currentDateTime)"
        let xslCode = "\(prefix)\n<!-- \(header) -->\n\(compiledXSL)"

        guard !compiledXSL.isEmpty else {
            let alert = NSAlert()
            alert.messageText = "Empty XSL"
            alert.informativeText = "Cannot save an empty XSL file."
            alert.alertStyle = .warning
            alert.runModal()
            return
        }

        do {
            let savePanel = NSSavePanel()
            savePanel.isFloatingPanel = true
            savePanel.nameFieldStringValue = "\(fileNameWithoutExtension).xsl"

            let res = savePanel.runModal()
            if res == NSApplication.ModalResponse.OK {
                let url = savePanel.url!
                try xslCode.write( to: url, atomically: false, encoding: String.Encoding.utf8 )
            }
        } catch {
            let alert = NSAlert()
            alert.messageText = "Error Saving XSL"
            alert.informativeText = "Failed to save XSL file: \(error.localizedDescription)"
            alert.alertStyle = .critical
            alert.runModal()
        }
    }

    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Main view
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

    var body: some View {
        HStack {
            HStack {
                Button( "Save" ) {
                    Task {
                        NSApp.sendAction(#selector(NSDocument.save(_:)), to: nil, from: nil)
                        isSourceDirty = false
                    }
                }
                .disabled( !isSourceDirty )
                
                Button( "Save as" ) {
                    Task {
                        NSApp.sendAction(#selector(NSDocument.saveAs(_:)), to: nil, from: nil)
                    }
                }
            }
            
            HStack {
                Spacer()
                Toggle( "Line Numbers", isOn: $insertLineNumbers )
                Toggle( "Show Undefined", isOn: $showUndefined )
                Toggle( "Default XMLNS", isOn: $defaultXMLNS )
                Toggle( "Auto Compile", isOn: $autoCompile )
                Spacer()
            }
            
            HStack {
                Button( "Save XSL" ) {
                    Task {
                        saveXSL( listingText )
                    }
                }
            }
        }
    }
}

// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// MARK: - Preview
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

#Preview {
    ToolBarView( sourceDocument: .constant( RexselSourceStruct() ),
                 insertLineNumbers: .constant( false ),
                 showUndefined: .constant( false ),
                 autoCompile: .constant( false ),
                 defaultXMLNS: .constant( false ),
                 listingText: .constant( "" ),
                 isSourceDirty: .constant( false ) )
    .frame( minWidth: 750 )
    .padding(.all)
}

