//
//  RexselEditorApp.swift
//  RexselEditor
//
//  Created by Hugh Field-Richards on 30/10/2024.
//  Copyright 2025 Hugh Field-Richards. All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  https://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.

import SwiftUI

// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// MARK: - RexselEditorApp Structure
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

@main
struct RexselEditorApp: App {
   
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // MARK: - KeyProperties
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   
   @AppStorage( KeyProperties.initialNewDocumentTextKey ) var initialNewDocumentText: String = ""
   @AppStorage( KeyProperties.unCompilerDateFormatKey ) var uncompilerDateFormat: String = ""
   @AppStorage( KeyProperties.uncompilerScriptKey ) var uncompilerScript: String = ""
   @AppStorage( KeyProperties.rexselFileExtKey ) var rexselFileExt: String = "rxsl"
   @AppStorage( KeyProperties.xsltProcessorKey ) var xsltProcessor: String = "/usr/bin/xsltproc"
   
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // MARK: - Instance Properties
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   
   /// Time stamp
   var currentDateTime: String {
      let formatter = DateFormatter()
      formatter.dateFormat = uncompilerDateFormat
      return formatter.string(from: Date())
   }
   
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // MARK: - Instance Functions
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   
   func getAppVersion() -> String {
      if let bundle = Bundle.main.infoDictionary {
         if let versionString = bundle[ "CFBundleShortVersionString" ] as? String {
            return versionString
         }
      }
      return "Unknown"
   }
   
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   
   func importXSLFile() {
      let panel = NSOpenPanel()
      panel.title = "Import XSL File"
      panel.allowedContentTypes = [.xml, .init(filenameExtension: "xsl") ?? .xml, .init(filenameExtension: "xslt") ?? .xml]
      panel.allowsMultipleSelection = false
      panel.canChooseDirectories = false
      panel.canChooseFiles = true
      
      panel.begin { response in
         if response == .OK, let url = panel.url {
            Task {
               await self.performXSLImport(from: url)
            }
         }
      }
   }
   
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   
   @MainActor
   private func performXSLImport(from sourceURL: URL) async {
      
//      if !FileManager.default.fileExists( atPath: uncompilerScript ) {
//         guard let bundledUncompilerScript = Bundle.main.path( forResource: "xsl2rexsel", ofType: "xsl" ) else {
//            showImportError( "Cannot find uncompile stylesheet (xsl2rexsel.xsl)")
//            return
//         }
//         uncompilerScript = bundledUncompilerScript
//      }
//      
  
      guard FileManager.default.fileExists( atPath: uncompilerScript ) else {
         showImportError( "Cannot find uncompile stylesheet \(uncompilerScript)")
         return
      }
      
      let fileName = sourceURL.deletingPathExtension().lastPathComponent
      //      let inputFullFileName = sourceURL.lastPathComponent
      let outputFileName = "\(fileName).\(rexselFileExt)"
      
      let uncompileScriptURL = URL( fileURLWithPath: uncompilerScript )
      
      let task = Process()
      task.launchPath = "/usr/bin/xsltproc"
      
      task.arguments = [ uncompileScriptURL.absoluteString, sourceURL.absoluteString ]
      let pipe = Pipe()
      let errorPipe = Pipe()
      task.standardOutput = pipe
      task.standardError = errorPipe
      
      do {
         try task.run()
         task.waitUntilExit()
         
         let data = pipe.fileHandleForReading.readDataToEndOfFile()
         let errorData = errorPipe.fileHandleForReading.readDataToEndOfFile()
         
         if !errorData.isEmpty {
            if let errorMessage = String(data: errorData, encoding: .utf8) {
               showImportError("XSLT Processing Error: \(errorMessage)")
               return
            }
         }
         
         guard var rexselScript = String(data: data, encoding: .utf8) else {
            showImportError("Error: Could not decode XSLT output")
            return
         }
         
         if !rexselScript.isEmpty {
            // Add timestamp comment to the converted content
            rexselScript = "// Produced on \(currentDateTime)\n\(rexselScript)"
            
            // Create a save panel to let user choose where to save the converted file
            let savePanel = NSSavePanel()
            savePanel.title = "Save Imported Rexsel File"
            savePanel.allowedContentTypes = [.init(filenameExtension: "rxsl") ?? .plainText]
            savePanel.nameFieldStringValue = outputFileName
            
            savePanel.begin { response in
               if response == .OK, let saveURL = savePanel.url {
                  do {
                     try rexselScript.write(to: saveURL, atomically: true, encoding: .utf8)
                     
                     // Open the converted file in a new document window
                     NSDocumentController.shared.openDocument(withContentsOf: saveURL, display: true) { _, _, _ in }
                     
                  } catch {
                     self.showImportError("Could not save converted file: \(error.localizedDescription)")
                  }
               }
            }
         }
         
      } catch {
         showImportError("Error running xsltproc: \(error)")
      }
   }
   
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   
   private func showImportError(_ message: String) {
      let alert = NSAlert()
      alert.messageText = "Import Error"
      alert.informativeText = message
      alert.alertStyle = .warning
      alert.addButton(withTitle: "OK")
      alert.runModal()
   }
   
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // MARK: - Initialisation Methods
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   
   init() {
      primePlistFromDefaultsIfRequired()
      setGlobals()
   }
   
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   
   func primePlistFromDefaultsIfRequired() {
      
      // Get the info.plist comtents which represent the default
      if let path = Bundle.main.path(
         forResource: rexselEditorPropertiesName,
         ofType: "plist"
      ) {
         defaultResourceFileDictionary = NSDictionary(contentsOfFile: path)
      } else {
         fatalError("\(rexselEditorPropertiesName).plist not found")
      }
      
      // Override with values from HVAccountsProperties.plist where mappings exist
      if let defaultDictionaryContent = defaultResourceFileDictionary {
         // defaultDictionaryContent has all the values in the default list
         // which in stored is HVAccountsProperties.plist in the app resources.
         //         logger.debug(defaultDictionaryContent.description)
         
         for (userDefaultsKey, _ /*userDefaultsValue*/) in defaultDictionaryContent {
            // All value in this plist are Strings.
        //    let valueFromAppDefaults: String = userDefaultsValue as! String
            let plistKey: String = userDefaultsKey as! String
            
            // If value from current user preferences is blank then update from default
            let valueFromCurrentPreferences = UserDefaults.standard.string(forKey: plistKey) ?? ""
            let valueFromUsersDefaults = defaultDictionaryContent[plistKey] as! String
            // Only update if there is no entry in the user default.
            if valueFromCurrentPreferences.isEmpty {
               UserDefaults.standard.set(
                  valueFromUsersDefaults,
                  forKey: plistKey
               )
//               print( "Priming user default value for '\(plistKey)' from Info.plist: '\(valueFromAppDefaults)'" )
            }
         }
      }
   }
   
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   
   func setGlobals() {
      initialNewDocumentText =  UserDefaults.standard.string(forKey: KeyProperties.initialNewDocumentTextKey ) ?? ""
      uncompilerDateFormat =  UserDefaults.standard.string(forKey: KeyProperties.unCompilerDateFormatKey ) ?? ""
      uncompilerScript =  UserDefaults.standard.string(forKey: KeyProperties.uncompilerScriptKey ) ?? ""
   }
   
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // MARK: - Main Scene
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   
   var body: some Scene {
      DocumentGroup( newDocument: RexselSourceStruct() ) { file in
         ContentView( sourceDocument: file.$document )
      }
      .commands {
         // Turn on find/replace etc. for panels.
         TextEditingCommands()
         CommandGroup(replacing: .importExport) {
            Button("Import XSL File…") {
               importXSLFile()
            }
            .keyboardShortcut("i", modifiers: [.command, .shift])
            Divider()
         }
         CommandGroup(replacing: .appInfo) {
            Button("About Rexsel") {
               NSApplication.shared
                  .orderFrontStandardAboutPanel (
                     options: [
                        .applicationName: "Rexsel",
                        .applicationVersion: "\(getAppVersion())"
                     ]
                  )
            }
         }
      }
      
      Settings {
         SettingsView()
      }
   }
}
