//
//  Utilities.swift
// HV Accounts
//
//  Created by Hugh Field-Richards on 17/02/2022.
//  Copyright 2022 Hugh Field-Richards. All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  https://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.

import Foundation
import Cocoa
import AppKit
import RegexBuilder
import WebKit
import SwiftUI

// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
//
/// The state of the navigation buttons.
///
/// -  _enabled_  :   Button is enabled (normal unselected state)
/// -  _selected_ :  Button is being pressed
/// -  _disabled_ :  Button is disabled (greyed out)

enum ButtonStateEnum {
   case enabled
   case selected
   case disabled
}

// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

struct CustomButtonStyle: ButtonStyle {
   
   private var buttonColor: Color = Color( red: 0.35, green: 0.35, blue: 0.35 )
   
   private var buttonColorSelected: Color = Color( red: 0.5, green: 0.5, blue: 0.5 )
   
   private var buttonColorDisabled: Color = Color( red: 0.25, green: 0.25, blue: 0.25 )
   
   private var buttonShape: RoundedRectangle = RoundedRectangle( cornerRadius: 5 )
   
   var buttonWidth: CGFloat = 250
   
   var buttonHeight: CGFloat = 22
   
   var buttonState: ButtonStateEnum = .enabled
   
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   
   func makeBody( configuration: Configuration ) -> some View {
      
      let isPressed = configuration.isPressed
      
      var backGroundColour: Color {
         if isPressed && buttonState == .enabled { return buttonColorSelected }
         if buttonState == .disabled { return buttonColorDisabled }
         return buttonColor
      }
      
      var textColor: Color {
         if buttonState == .disabled { return Color.gray }
         return .white
      }
      
      return configuration.label
         .buttonStyle( .plain )
         .frame( width: buttonWidth, height: buttonHeight )
         .background( backGroundColour )
         .foregroundColor( textColor )
         .containerShape( buttonShape )
         .opacity( buttonState == .disabled ? 0.6 : 1.0 )
   }
}

// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
//
/// Get the reference (URL) of a folder which the user
/// can choose.
///
/// - Parameters:
///    - initialFile: the suggested folder
/// - Returns: The URL of the selected file

func getFolderStartingWith( _ initialFolder: String ) -> URL?
{
   // Set up a dialog box to get spreadsheet filename
   let panel = NSOpenPanel()
   panel.allowsMultipleSelection = false
   panel.canChooseDirectories = true
   panel.canChooseFiles = true
   panel.isFloatingPanel = false
   
   // Set up the folder for importing (can be changed in the open window)
   let defaultDataPath = NSString( string: initialFolder ).deletingLastPathComponent as String
   panel.directoryURL = URL( fileURLWithPath: defaultDataPath )
   
   let clicked = panel.runModal()
   if clicked == NSApplication.ModalResponse.OK {
      let files = panel.urls
      // Take first file selected (does not allow multiple selection)
      let fileURL = (files[0] )
      return fileURL
   }
   return nil
}

