//
//  ContentView.swift
//  RexselEditor
//
//  Created by Hugh Field-Richards on 30/10/2024.
//  Copyright 2025 Hugh Field-Richards. All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  https://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
//
// This is the main view for the app.

import SwiftUI
import RexselKernel
import LanguageSupport
import CodeEditorView

struct ContentView: View {
      
    @Binding var sourceDocument: RexselSourceStruct
    
    /// The listing text (XSL) to be placed in the top right panel.
    @State var listingText = ""
    
    /// The symbol table and errors text to be placed in the bottom right panel.
    @State var symbolsAndErrorsText = ""
  
    /// A list of the error lines.
    @State var errorLines = [Int]()
  
    /// A set of error messages.
    @State var errorMessages: Set<LanguageSupport.TextLocated<LanguageSupport.Message>> = Set()
    
     /// Include the source line numbers in the XSL output.
    @State var insertLineNumbers: Bool = false
    
    /// Show any undefined symbols as well as normal errors.
    /// This is used when Rexsel/XSL modules are used. It helps
    /// by only showing syntax errors.
    @State var showUndefined: Bool = false
    
    /// Allow auto compile (this can be turned off when large files are used).
    @State var autoCompile: Bool = false
    
    /// Inhibit the explicit use of XSL namespaces throughout
    /// the XSL elements.
    ///
    /// In other words use
    /// ```
    ///    xmlns="http://www.w3.org/1999/XSL/Transform"
    /// ```
    /// rather than
    /// ```
    ///    xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    ///    ```
    @State var defaultXMLNS: Bool = false
    
    /// Time in seconds before needing recompile.
    @State var delayTimeBeforeCompiling = 1

    /// Timer count.
    @State var delayTime = 0

    /// The compiler associated with this source.
    var compiler = RexselKernel.sharedInstance
    
    /// The wrapper for the compiler
    @State var thisCompiler: RunCompiler?
    
    /// Does the source document need saving?
    @State var isSourceDirty: Bool = false
    
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Structure Instance methods
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

    func invokeCompiler( withDelayInSeconds initialTime: Int = 0, updateOnly: Bool = false ) {
        delayTime = initialTime
        Timer.scheduledTimer( withTimeInterval: 1.0, repeats: true )  { timer in
            delayTime -= 1
            if ( delayTime <= 0 ) {
                if thisCompiler == nil {
                    thisCompiler = RunCompiler( with: sourceDocument.text,
                                                using: compiler,
                                                lineNumbers: insertLineNumbers,
                                                showUndefined: showUndefined,
                                                useDefaultXMLNS: defaultXMLNS )
                }
                
                if let result = thisCompiler?.run( with: sourceDocument.text,
                                                   lineNumbers: insertLineNumbers,
                                                   showUndefined: showUndefined,
                                                   useDefaultXMLNS: defaultXMLNS ) {
                    errorLines = result.errorLines.sorted()
                    errorMessages = result.errorMessages
                    listingText = result.codeListing
                    symbolsAndErrorsText = result.statusListing
                }
                timer.invalidate()
            }
        }
    }
    
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Main view
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

    var body: some View {
        VStack() {
            Spacer()
            VStack {
                HSplitView {
                    VStack {
                        VersionView()
                        SourceView( sourceDocument: $sourceDocument,
                                    errorMessages: $errorMessages,
                                    errorLines: $errorLines,
                                    insertLineNumbers: $insertLineNumbers,
                                    showUndefined: $showUndefined,
                                    defaultXMLNS: $defaultXMLNS )
                        .onAppear() {
                            invokeCompiler( withDelayInSeconds: delayTimeBeforeCompiling )
                        }
                        .onChange( of: sourceDocument.text ) {
                            isSourceDirty = true
                            if autoCompile {
                                invokeCompiler( withDelayInSeconds: delayTimeBeforeCompiling )
                            }
                        }
                    } .frame( minWidth: 600 )
                        .padding(EdgeInsets(top: 0, leading: 0, bottom: 0, trailing: 15))
                    VSplitView {
                        HStack {
                            VStack {
                                ListingView( listingText: $listingText )
                                    .frame( minHeight: 100 )
                                Spacer()
                                HStack {
                                    Spacer()
                                    Button( "Generate XSL" ) {
                                        Task {
                                            invokeCompiler( updateOnly: true )
                                        }
                                    }
                                    .padding([.bottom], 10)
                                    .disabled( autoCompile )
                                    Spacer()
                                }
                            }
                        }
                        SymbolsView( symbolsAndErrorsText: $symbolsAndErrorsText )
                            .frame( minHeight: 100 )
                    } .frame( minWidth: 500 )
                } .padding([.leading, .trailing, .top])
                
                Divider()
                HStack {
                    ToolBarView( sourceDocument: $sourceDocument,
                                 insertLineNumbers: $insertLineNumbers,
                                 showUndefined: $showUndefined,
                                 autoCompile: $autoCompile,
                                 defaultXMLNS: $defaultXMLNS,
                                 listingText: $listingText,
                                 isSourceDirty: $isSourceDirty )
                    .onChange( of: insertLineNumbers ) {
                        if autoCompile {
                            invokeCompiler( updateOnly: true )
                        }
                    }
                    .onChange( of: showUndefined ) {
                        if autoCompile {
                            invokeCompiler( updateOnly: true )
                        }
                    }
                    .onChange( of: defaultXMLNS ) {
                        if autoCompile {
                            invokeCompiler( updateOnly: true )
                        }
                    }
                } .padding([.leading, .trailing, .bottom])
            }
        }
        .frame( minWidth: 1500, minHeight: 600 )
        .background( Color.gray.opacity(0.1) )

    }
}

// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// MARK: - Preview
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

#Preview {
    let someSourceText = """
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
//
// Error Page to HTML Rexsel Transform
//
// Author:
//    Name  : Hugh Field-Richards
//    Email : hsfr@hsfr.org.uk
//
"""
    
    let sourceDocument = RexselSourceStruct( text: someSourceText )
    let listingsText = "Listings go here"
    let symbolsText = "Symbols and errors go here"
    
    ContentView( sourceDocument: .constant(sourceDocument),
                 listingText: listingsText,
                 symbolsAndErrorsText: symbolsText )
    
}

