//
//  ListingView.swift
//  RexselEditor
//
//  Created by Hugh Field-Richards on 30/10/2024.
//  Copyright 2025 Hugh Field-Richards. All rights reserved.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  https://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
//
// This is the editor panel view for the app.

import SwiftUI
import LanguageSupport
import CodeEditorView
import RexselKernel

/// An enumeration of one to satisfy the CodeEditor.
enum Language: Hashable {
    case swift

    var configuration: LanguageConfiguration {
        switch self {
            case .swift: .swift()
        }
    }
}

// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// MARK: - Source View Structure
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

struct SourceView: View {
    
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Binding properties

   /// The actual source to be displayed (.text).
    @Binding var sourceDocument: RexselSourceStruct
    
     /// The actual source to be displayed (.text).
    @Binding var errorMessages: Set<LanguageSupport.TextLocated<LanguageSupport.Message>>
    
    /// A list of the error lines.
    @Binding var errorLines: [Int] {
        mutating didSet {
            if errorLines.isEmpty {
                currentError = 0
            }
        }
    }
    
    /// Include the source line numbers in the XSL output.
    @Binding var insertLineNumbers: Bool
    
    /// Show the undefined symbols. Normally inhibited to only show syntax errors.
    @Binding var showUndefined: Bool
    
    /// Inhibit the explicit use of XSL namespaces throughout
    /// the XSL elements.
    ///
    /// In other words use
    /// ```
    ///    xmlns="http://www.w3.org/1999/XSL/Transform"
    /// ```
    /// rather than
    /// ```
    ///    xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    /// ```
    @Binding var defaultXMLNS: Bool
    
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - State properties

    /// The listing text (XSL) to be placed in the top right panel.
    @State var listingText = ""

    /// The symbol table and errors text from the compiler to be
    /// placed in the bottom right panel.
    @State var symbolsAndErrorsText = ""
    
    /// The colour scheme that the editor uses: Light, Dark etc.
    @Environment(\.colorScheme) private var colorScheme: ColorScheme
    
    // NB: Writes to a @SceneStorage backed variable are sometimes (always?)
    // not available in the update cycle where the update occurs, but only one
    // cycle later. That can lead to back and forth bouncing values and other
    // problems in views that take multiple bindings as arguments.
    @State private var editPosition: CodeEditor.Position = .init()

    @SceneStorage( "editPosition" ) private var editPositionStorage: CodeEditor.Position?

    /// The error messages to be display inline.
    @State var messages: Set<TextLocated<Message>> = Set()
    
    @State private var language:         Language      = .swift
    @State private var theme:            ColorScheme?  = nil
    @State private var showMessageEntry: Bool          = false
    @State private var showMinimap:      Bool          = true
    @State private var wrapText:         Bool          = true

    @FocusState private var editorIsFocused: Bool
    
    @State var currentError: Int = 0

    @State var previousError: Int = 0
    
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Instance properties
    
    /// The total lines in the source for display at the bottom of the text area.
    private var totalLines: Int {
        return sourceDocument.text.numberOfLines
    }
    
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // MARK: - Main view
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
    // -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

    var body: some View {
        VStack {
            HStack {
                VStack {
                    
                    CodeEditor( text: $sourceDocument.text,
                                position: $editPosition,
                                messages: $errorMessages,
                                language: language.configuration
                    )
                    .environment( \.codeEditorTheme,
                                   (theme ?? colorScheme) == .dark ? Theme.defaultDark : Theme.defaultLight)
                    .environment( \.codeEditorLayoutConfiguration,
                                   CodeEditor.LayoutConfiguration(showMinimap:  showMinimap, wrapText: wrapText) )
                    .focused($editorIsFocused)
                    
                    Spacer()
                    HStack {
                        Spacer()
                        HStack {
                            Picker("Theme: ", selection: $theme) {
                                Text("Default").tag(nil as ColorScheme?)
                                Text("Light").tag(ColorScheme.light as ColorScheme?)
                                Text("Dark").tag(ColorScheme.dark as ColorScheme?)
                            } .frame( width: 200 )
                        }
                        HStack {
                            Spacer()
                            Toggle("Minimap", isOn: $showMinimap)
                                .toggleStyle(.checkbox)
                            
                            Toggle("Wrap", isOn: $wrapText)
                                .toggleStyle(.checkbox)
                            Spacer()
                            Text( "Total lines: \(totalLines)" )
                        }
                        Spacer()
                    }
                }
            }
        }
    }
}

// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// MARK: - Preview
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*

#Preview {
    let someText = """
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
// -*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
//
// Error Page to HTML Rexsel Transform
//
// Author:
//    Name  : Hugh Field-Richards
//    Email : hsfr@hsfr.org.uk
//
"""

    let sourceDocument = RexselSourceStruct( text: someText )
    let errorMessages: Set<LanguageSupport.TextLocated<LanguageSupport.Message>> = Set()
    let errorLines = [Int]()
    
    SourceView( sourceDocument: .constant(sourceDocument),
                errorMessages: .constant( errorMessages ),
                errorLines: .constant( errorLines ),
                insertLineNumbers: .constant( true ),
                showUndefined: .constant( true ),
                defaultXMLNS: .constant( true )
    )
}

